# IMPORT THE OR-TOOLS CONSTRAINT SOLVER
from ortools.constraint_solver import pywrapcp
import sys
import json
import math

# LOAD PROBLEM DATA
if len(sys.argv) != 2:
    print 'Usage: python %s <data file>' % sys.argv[0]
    sys.exit()
else:
    fname = sys.argv[1]

with open(fname) as fin:
    data = json.load(fin)

# Cache some useful data
services = data['services']
nservices = len(services)
nservers = data['nservers']
cap_cpu = data['cap_cpu']

# split services into individual VMs
vm_svc = [] # service idx, for each VM
vm_cpu = [] # CPU requirement, for each VM
for k, svc in enumerate(services):
    vm_num = svc['vm_num']
    vm_svc += [k] * vm_num
    vm_cpu += [svc['vm_cpu']] * vm_num
# Total number of virtual machines
nvm = len(vm_svc)

# Build solver instance
slv = pywrapcp.Solver('vm-reassignment')

# Cost variable (number of used services)
z = slv.IntVar(0, nservers, 'z')

# One variable for each VM
x = [slv.IntVar(0, nservers-1, 'x_%d' % i) for i in range(nvm)]

# VMs within the same service should go on different servers
for i in range(nvm-1):
    for j in range(i+1, nvm):
        if vm_svc[i] == vm_svc[j]:
            slv.Add(x[i] != x[j])

# CPU capacity constraints
srv_usgs = []
for j in range(nservers):
    # Obtain an expression for the total CPU requirement
    usg = sum(v * (x[i] == j) for i, v in enumerate(vm_cpu))
    # Store the expression that corresponds to the server usae
    srv_usgs.append(usg)
    # Post CPU capacity constraint
    slv.Add(usg <= cap_cpu)

# Cost definition (similar to the map coloring problem)
slv.Add(z == sum((srv_usgs[j] > 0 for j in range(nservers))))

# DEFINE THE SEARCH STRATEGY
decision_builder = slv.Phase(x,
                             slv.INT_VAR_DEFAULT,
                             slv.INT_VALUE_DEFAULT)

# INIT THE SEARCH PROCESS
time_limit = 30000
search_monitors = [slv.Minimize(z, 1),
                   slv.TimeLimit(time_limit)]
# init search
slv.NewSearch(decision_builder, search_monitors)

# SEARCH FOR A FEASIBLE SOLUTION
zbest = None
while slv.NextSolution():
    #
    # PRINT SOLUTION
    #
    print '--- Solution found, time: %.3f (sec), branches: %d' % \
                (slv.WallTime()/1000.0, slv.Branches())
    print '--- z: %d' % z.Value()
    print '--- x:', ', '.join('%3d' % var.Value() for var in x)

    # NOTE: srv_usgs[i] corresponds to a sum, i.e. to an _expressions_. Expressions
    # in or-tool do not have a "Value()" method, but they do have a "Min()" and
    # "Max()" method to access the bounds of their domain
    cpu_req = [srv_usgs[j].Min() for j in range(nservers)]
    print '--- usg:', ', '.join('%3d' % v for v in cpu_req)
    print

    # STORE SOLUTION VALUE
    zbest = z.Value()

# END THE SEARCH PROCESS
slv.EndSearch()

# obtain stats
branches, time = slv.Branches(), slv.WallTime()
# time capping
time = max(1, time)

# print stats
print '--- FINAL STATS'
if zbest == None:
    print '--- No solution found'
print '--- Number of branches: %d' % branches
print '--- Computation time: %.3f (sec)' % (time / 1000.0)
if slv.WallTime() > time_limit:
    print '--- Time limit exceeded'
