#
# IMPORT THE OR-TOOLS CONSTRAINT SOLVER
#
from ortools.constraint_solver import pywrapcp
import sys
import json

#
# Optimization methods
#

#
# A FUNCTION TO BUILD AND SOLVE A MODEL
# time_limit: if None, not time limit is employed. If integer, a time limit is
#             enforced, but optimality is no longer guaranteed!
# lb:         lower bound
# ub:         upper bound
# 
# If both 'lb' and 'ub' are None, then Branch and bound is used for optimization
#
def solve_problem(data, time_limit = None, lb = None, ub = None):
    # Cache some useful data
    setups = data['setups']
    order_list = data['order_list']
    unit_list = data['unit_list']
    order_table = data['order_table']
    no = len(order_list)
    nu = len(unit_list)

    # Build solver instance
    slv = pywrapcp.Solver('production-scheduling')

    #
    # CREATE VARIABLES
    #

    # Compute a safe time horizon
    eoh = max(o['dline'] for o in order_list)
    # Cache the number of product types
    np = len(order_table)

    # A variable for each time unit
    x = [slv.IntVar(-1, np-1, 's_%d' % i) for i in range(eoh+1)]

    # Objective variable
    z = slv.IntVar(0, eoh, 'z')

    #
    # BUILD CONSTRAINTS AND ADD THEM TO THE MODEL
    #

    # Deadline constraint, version 1: using Count()
    for p in range(np):
        for t in range(eoh+1):
            if order_table[p][t] > 0:
                # Count the number of product of type 'p' that should be
                # produced up to time 't'
                req = sum(order_table[p][i] for i in range(t+1))
                # Post the deadline constraints
                slv.Add(slv.Count([x[i] for i in range(t+1)], p, slv.IntVar(req, eoh)))

    # Deadline constraint, version 2: using Count()
    # NOTE: for this problem, using the Distribute constraint provides no advantage in
    # terms of propagation. Therefore, it is better to stick with Count()
    # for t in range(eoh+1):
    #     if any(order_table[p][t] > 0 for p in range(np)):
    #             req = [sum(order_table[p][i] for i in range(t+1)) for p in range(np)]
    #             rvars = [slv.IntVar(req[p], eoh) for p in range(np)]
    #             xvars = [x[i] for i in range(t+1)]
    #             slv.Add(slv.Distribute(xvars, range(np), rvars))

    # Setup times
    for t in range(eoh):
        for p1, p2 in setups:
            slv.Add((x[t] == p1) <= (x[t+1] != p2))

    # z definition constraints
    # The makespan is the largest non-idle time point
    slv.Add(z == slv.Max([i * (x[i] != -1) for i in range(eoh+1)]))

    # Optional bounding constraints
    if lb != None:
        slv.Add(z >= lb)
    if ub != None:
        slv.Add(z <= ub)

    #
    # THOSE ARE THE VARIABLES THAT WE WANT TO USE FOR BRANCHING
    #
    all_vars = x

    # DEFINE THE SEARCH STRATEGY
    decision_builder = slv.Phase(all_vars,
                                 slv.INT_VAR_DEFAULT,
                                 slv.INT_VALUE_DEFAULT)

    # INIT THE SEARCH PROCESS

    # log monitor (just to have some feedback)
    # search_monitors = [slv.SearchLog(500000)]
    search_monitors = []
    # enforce a time limit (if requested)
    if time_limit:
       search_monitors.append(slv.TimeLimit(time_limit))
    # enable branch and bound
    if lb == None and ub == None:
        search_monitors.append(slv.Minimize(z, 1))

    # init search
    slv.NewSearch(decision_builder, search_monitors)

    # SEARCH FOR A FEASIBLE SOLUTION
    zbest = None
    while slv.NextSolution():
        #
        # PRINT SOLUTION
        #
        print 'SOL FOUND ====================================='
        print 'time: %.3f (sec), branches: %d' % (slv.WallTime()/1000.0, slv.Branches())
        print 'z: %d' % z.Value()
        def content(j, x):
            if x[j].Value() >= 0: return '%2d' % x[j].Value()
            return '--'
        print 'sol: ' + ' '.join(content(j, x) for j in range(eoh+1))
        print

        #
        # STORE SOLUTION VALUE
        #
        zbest = z.Value()

    # END THE SEARCH PROCESS
    slv.EndSearch()

    # print stats
    print 'FINAL STATS ==================================='
    if zbest == None:
        print '*** No solution found'
    branches, time = slv.Branches(), slv.WallTime()/1000.0
    print '*** Number of branches: %d' % branches
    print '*** Computation time: %.3f (sec)' % time
    if time_limit != None and slv.WallTime() > time_limit:
        print '*** Time limit exceeded'


    # Return the solution value
    return zbest, branches, time


# LOAD PROBLEM DATA
if len(sys.argv) != 2:
    print 'Usage: python %s <data file>' % sys.argv[0]
    sys.exit()
else:
    fname = sys.argv[1]

with open(fname) as fin:
    data = json.load(fin)



#
# CALL THE SOLUTION APPROACH
#
solve_problem(data, time_limit=None)
