#
# IMPORT THE OR-TOOLS CONSTRAINT SOLVER
#
from ortools.constraint_solver import pywrapcp
import sys
import json

#
# A FUNCTION TO BUILD AND SOLVE A MODEL
# time_limit: if None, not time limit is employed. If integer, a time limit is
#             enforced, but optimality is no longer guaranteed!
#
def solve_problem(data, time_limit = None):
    # Cache some useful data
    setups = data['setups']
    order_list = data['order_list']
    unit_list = data['unit_list']
    order_table = data['order_table']
    no = len(order_list)
    nu = len(unit_list)

    unit_list = sorted(unit_list, key=lambda u: u['dline'])

    # Build solver instance
    slv = pywrapcp.Solver('production-scheduling')

    #
    # CREATE VARIABLES
    #

    # Compute a safe time horizon
    eoh = max(o['dline'] for o in order_list)

    # A variable for each unit to be produced
    x = [slv.IntVar(0, eoh, 's_%d' % i) for i in range(nu)]

    # Objective variable
    z = slv.IntVar(0, eoh, 'z')

    #
    # BUILD CONSTRAINTS AND ADD THEM TO THE MODEL
    #

    # no parallel manufacturing
    slv.Add(slv.AllDifferent(x))

    # deadline constraints
    for i, unit in enumerate(unit_list):
        slv.Add(x[i] <= unit['dline'])

    # setup times
    for p1, p2 in setups:
        for i, u1 in enumerate(unit_list):
            for j, u2 in enumerate(unit_list):
                if i != j and u1['prod'] == p1 and u2['prod'] == p2:
                    slv.Add(x[j] != x[i] + 1)

    # z definition constraints
    slv.Add(z == slv.Max(x))

    #
    # THOSE ARE THE VARIABLES THAT WE WANT TO USE FOR BRANCHING
    #
    all_vars = x

    # DEFINE THE SEARCH STRATEGY
    decision_builder = slv.Phase(all_vars,
                                 slv.INT_VAR_DEFAULT,
                                 slv.INT_VALUE_DEFAULT)

    # INIT THE SEARCH PROCESS

    # log monitor (just to have some feedback)
    # search_monitors = [slv.SearchLog(500000)]
    search_monitors = [slv.Minimize(z, 1)]
    # enforce a time limit (if requested)
    if time_limit:
       search_monitors.append(slv.TimeLimit(time_limit))

    # init search
    slv.NewSearch(decision_builder, search_monitors)

    # SEARCH FOR A FEASIBLE SOLUTION
    zbest = None
    while slv.NextSolution():
        #
        # PRINT SOLUTION
        #
        print 'SOL FOUND ====================================='
        print 'time: %.3f (sec), branches: %d' % (slv.WallTime()/1000.0, slv.Branches())
        print 'z: %d' % z.Value()
        def content(j, x):
            for i, var in enumerate(x):
                if var.Value() == j: return '%2d' % i
            return '--'
        print 'sol: ' + ' '.join(content(j, x) for j in range(eoh+1))
        print

        #
        # STORE SOLUTION VALUE
        #
        zbest = z.Value()

    # END THE SEARCH PROCESS
    slv.EndSearch()

    # print stats
    print 'FINAL STATS ==================================='
    if zbest == None:
        print '*** No solution found'
    branches, time = slv.Branches(), slv.WallTime()/1000.0
    print '*** Number of branches: %d' % branches
    print '*** Computation time: %.3f (sec)' % time
    if time_limit != None and slv.WallTime() > time_limit:
        print '*** Time limit exceeded'

    # Return the solution value
    return zbest, branches, time


# LOAD PROBLEM DATA
if len(sys.argv) != 2:
    print 'Usage: python %s <data file>' % sys.argv[0]
    sys.exit()
else:
    fname = sys.argv[1]

with open(fname) as fin:
    data = json.load(fin)


#
# CALL THE SOLUTION APPROACH
#
solve_problem(data, time_limit=None)
